﻿using log4net;
using Microsoft.WindowsAzure.Storage;
using Microsoft.WindowsAzure.Storage.Table;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using VA.PPMS.IWS.Functions.Configuration.Interface;
using VA.PPMS.IWS.TableService.Interface;
using VA.PPMS.IWS.TableService.Interface.Entities;

namespace VA.PPMS.IWS.TableService
{
    public class TableService : ITableService
    {
        private readonly ILog _logger;
        private readonly IIwsConfiguration _configuration;

        public TableService(ILog logger, IIwsConfiguration configuration)
        {
            _logger = logger;
            _configuration = configuration;
        }

        public int BatchSize => 100;

        public int SuperBatchSize => 1000;

        public string StatusActive => "Active";

        public string StatusInactive => "Inactive";

        public async Task<CloudTable> GetTableRef()
        {
            try
            {
                _logger.Info("@@@@ INFO - Start GetTableRef  @@@@");

                _logger.Info("@@@@ INFO - End GetTableRef  @@@@");

                return await GetTableRef(CloudTableType.DefaultTableNameSetting);
            }
            catch (Exception ex)
            {
                _logger.Error("@@@@ ERROR - There was a problem with the GetTableRef @@@@", ex);
                throw;
            }
        }

        public async Task<CloudTable> GetTableRef(CloudTableType tableType)
        {
            try
            {
                string tableName;

                switch (tableType)
                {
                    case CloudTableType.ActivityTableNameSetting:
                        tableName = await GetActivityTableName();
                        break;
                    default:
                        tableName = await GetNpiStatusTableName();
                        break;
                }

                var storageAccount = await GetCnn();
                if (storageAccount == null) return null;

                var tableClient = storageAccount.CreateCloudTableClient();

                return tableClient.GetTableReference(tableName);

            }
            catch (Exception ex)
            {
                _logger.Error("@@@@ ERROR - There was a problem with the GetTableRef @@@@", ex);
                throw;
            }
        }

        public async Task CreateTable(CloudTableType tableType)
        {
            try
            {
                _logger.Info("@@@@ INFO - Start CreateTable  @@@@");

                var table = await GetTableRef(tableType);
                table.CreateIfNotExists();

                _logger.Info("@@@@ INFO - End CreateTable  @@@@");
            }
            catch (Exception ex)
            {
                _logger.Error("@@@@ ERROR - There was a problem with the CreateTable @@@@", ex);
                throw;
            }
        }

        public async Task SaveNpiStatusAsync(string npi, string npiType, string status)
        {
            try
            {
                _logger.Info("@@@@ INFO - Start SaveNpiStatusAsync  @@@@");

                await SaveNpiStatusAsync(new NpiStatusEntity(npi, npiType, status));

                _logger.Info("@@@@ INFO - End SaveNpiStatusAsync  @@@@");
            }
            catch (Exception ex)
            {
                _logger.Error("@@@@ ERROR - There was a problem with the SaveNpiStatusAsync @@@@", ex);
                throw;
            }
        }

        public async Task SaveNpiStatusBatchAsync(IList<NpiStatusEntity> list)
        {
            try
            {
                _logger.Info("@@@@ INFO - Start SaveNpiStatusBatchAsync  @@@@");

                var keyList = new List<NpiStatusBatch>();
                NpiStatusBatch targetBatch;

                // Split the list into smaller batches by key value
                foreach (var item in list)
                {
                    targetBatch = keyList.FirstOrDefault(b => b.Key == item.Key);
                    if (targetBatch == null)
                    {
                        targetBatch = new NpiStatusBatch(item);
                        keyList.Add(targetBatch);
                    }
                    else
                    {
                        targetBatch.Batch.Add(item);
                    }
                }

                // Submit each batch
                foreach (var item in keyList)
                {
                    await ProcessBatchSave(item.Batch);
                }

                _logger.Info("@@@@ INFO - End SaveNpiStatusBatchAsync  @@@@");
            }
            catch (Exception ex)
            {
                _logger.Error("@@@@ ERROR - There was a problem with the SaveNpiStatusBatchAsync @@@@", ex);
                throw;
            }
        }

        private async Task ProcessBatchSave(IList<NpiStatusEntity> list)
        {
            if (list.Count > BatchSize)
            {
                IEnumerable<NpiStatusEntity> working;
                int i = 0;

                while (i < list.Count)
                {
                    working = list.Skip(i).Take(BatchSize);
                    i += BatchSize;
                    await TableBatchSave(working);
                }
            }
            else
            {
                await TableBatchSave(list);
            }
        }

        private async Task TableBatchSave(IEnumerable<NpiStatusEntity> list)
        {
            if (list.Count() > 100)
                throw new InvalidOperationException("Unable to process more than 100 NPI status records at a time.");

            var operation = new TableBatchOperation();

            // Add list to operations
            foreach (var item in list)
            {
                operation.InsertOrReplace(item);
            }

            var table = await GetTableRef();
            await table.ExecuteBatchAsync(operation);
        }

        public async Task SaveNpiStatusAsync(NpiStatusEntity npiStatus)
        {
            try
            {
                _logger.Info("@@@@ INFO - Start SaveNpiStatusAsync  @@@@");

                var insertOperation = TableOperation.InsertOrReplace(npiStatus);

                await ExecuteBatchAsync(CloudTableType.DefaultTableNameSetting, insertOperation);

                _logger.Info("@@@@ INFO - End SaveNpiStatusAsync  @@@@");
            }
            catch (Exception ex)
            {
                _logger.Error("@@@@ ERROR - There was a problem with the SaveNpiStatusAsync @@@@", ex);
                throw;
            }
        }

        public async Task<NpiStatusEntity> ReadNpiStatusAsync(string npi, string status)
        {
            try
            {
                _logger.Info("@@@@ INFO - Start ReadNpiStatusAsync  @@@@");

                var retrieveOperation = TableOperation.Retrieve<NpiStatusEntity>(status, npi);
                var table = await GetTableRef();
                var retrievedResult = await table.ExecuteAsync(retrieveOperation);

                if (retrievedResult.Result != null)
                {
                    return (NpiStatusEntity)retrievedResult.Result;
                }

                _logger.Info("@@@@ INFO - End ReadNpiStatusAsync  @@@@");

                return null;
            }
            catch (Exception ex)
            {
                _logger.Error("@@@@ ERROR - There was a problem with the ReadNpiStatusAsync @@@@", ex);
                throw;
            }
        }

        public async Task ClearNpiStatusAsync()
        {
            try
            {
                _logger.Info("@@@@ INFO - Start ClearNpiStatusAsync  @@@@");

                var table = await GetTableRef();
                if (table == null) return;

                // Delete all items from the table
                var query = new TableQuery<NpiStatusEntity>();
                var i = 1;

                var aBatch = new TableBatchOperation();
                var iBatch = new TableBatchOperation();

                foreach (var item in table.ExecuteQuery(query))
                {
                    if (item.PartitionKey == "Active") aBatch.Delete(item);
                    else iBatch.Delete(item);

                    if (i % BatchSize == 0)
                    {
                        if (i % BatchSize * 10 == 0) _logger.Info($"> Deleting batch [{i}]");

                        if (aBatch.Count > 0) await ExecuteBatchAsync(aBatch);

                        if (iBatch.Count > 0) await ExecuteBatchAsync(iBatch);

                        aBatch.Clear();
                        iBatch.Clear();

                        await GetTableRef();
                    }
                    i++;
                }

                if (aBatch.Count > 0) await ExecuteBatchAsync(aBatch);
                if (iBatch.Count > 0) await ExecuteBatchAsync(iBatch);

                _logger.Info("@@@@ INFO - End ClearNpiStatusAsync  @@@@");
            }
            catch (Exception ex)
            {
                _logger.Error("@@@@ ERROR - There was a problem with the ClearNpiStatusAsync @@@@", ex);
                throw;
            }
        }

        public async Task SetActivity(ActivityHistoryEntity activity)
        {
            try
            {
                _logger.Info("@@@@ INFO - Start SetActivity  @@@@");

                var insertOperation = TableOperation.InsertOrReplace(activity);
                await ExecuteBatchAsync(CloudTableType.ActivityTableNameSetting, insertOperation);

                _logger.Info("@@@@ INFO - End SetActivity  @@@@");
            }
            catch (Exception ex)
            {
                _logger.Error("@@@@ ERROR - There was a problem with the SetActivity @@@@", ex);
                throw;
            }
        }

        public async Task<ActivityHistoryEntity> GetActivity(ActivityHistoryEntity.ActivityType activityType)
        {
            try
            {
                _logger.Info("@@@@ INFO - Start GetActivity  @@@@");

                var retrievedResult = await GetCurrentActivityHistory(activityType);

                // Return the result
                if (retrievedResult != null)
                {
                    _logger.Info("@@@@ INFO - End GetActivity  @@@@");
                    return retrievedResult;
                }

                var now = DateTime.Today;
                var activity = new ActivityHistoryEntity(activityType, new DateTime(now.Year, now.Month, 1));

                await CreateTable(CloudTableType.ActivityTableNameSetting);
                await SetActivity(activity);

                _logger.Info("@@@@ INFO - End GetActivity  @@@@");

                return activity;
            }
            catch (Exception ex)
            {
                _logger.Error("@@@@ ERROR - There was a problem with the GetActivity @@@@", ex);
                throw;
            }
        }

        private async Task<ActivityHistoryEntity> GetCurrentActivityHistory(ActivityHistoryEntity.ActivityType activityType)
        {
            var retrieveOperation = TableOperation.Retrieve<ActivityHistoryEntity>(activityType.ToString(), activityType.ToString());

            var table = await GetTableRef(CloudTableType.ActivityTableNameSetting);
            var retrievedResult = await table.ExecuteAsync(retrieveOperation);

            return (ActivityHistoryEntity) retrievedResult.Result;
        }

        private async Task<CloudStorageAccount> GetCnn()
        {
            return CloudStorageAccount.Parse(await GetStorageConnection());
        }

        private async Task ExecuteBatchAsync(CloudTableType tableType, TableOperation op)
        {
            var table = await GetTableRef(tableType);
            await table.ExecuteAsync(op);
        }

        private async Task ExecuteBatchAsync(TableBatchOperation op)
        {
            var table = await GetTableRef();
            await table.ExecuteBatchAsync(op);
        }

        private async Task<string> GetStorageConnection()
        {
            return await _configuration.GetConnectionAsync();
        }

        private async Task<string> GetActivityTableName()
        {
            return await _configuration.GetActivityHistoryTableNameAsync();
        }

        private async Task<string> GetNpiStatusTableName()
        {
            return await _configuration.GetNpiStatusTableNameAsync();
        }
    }
}